clear;close all;clc


%%%%%%%%%%%%%%% Load GRACE data and time

ncfile = 'GRAVIS-3_2002095-2023059_COSTG_0100_AIS_GRID_TUD_0003.nc';
dm = ncread(ncfile, 'dm');  
dm = dm(:,:,1:216);  % limit to Dec 2022
x = ncread(ncfile, 'x');  
y = ncread(ncfile, 'y'); 
t_yr=ncread(ncfile,'time_dec');  
t_yr = t_yr(1:216);  % limit to Dec 2022

cmap = cptcmap('curl','ncol',23);
cmap = flipud(cmap);

%%%%%%%%%%   load antarctic coast line %%%%%%%%%%%%%%%%%%%%%%%%
cl = load('moa2014_grounding_line_v01.gmt');
c1=cl(:,1);c2=cl(:,2);

%%%%%%%%%%%%%%  load Antarctic basin shape fine %%%%%%%%%%%%%%

basins = shaperead('ANT_Basins_IMBIE2_v1.6.shp');




%%%%%%% Smooth GRACE time to remove short-term noise %%%%%%%%%%

dm = smoothdata(dm, 3, "movmedian", 7/12, 'SamplePoints', t_yr); %%% 7-month window

%%%%%%%%%%%%%%%%%%%%%%% Build quadratic design matrix:%%%%%%%%%%%%%%%% 

t = t_yr - mean(t_yr); 

X = [ones(length(t_yr),1), t, t.^2];
            

%%%%%%%%  Initialize outputs

[nx, ny, ~]= size(dm);

trend = nan(nx, ny);            % linear slope (β₁)
acceleration = nan(nx, ny);      % 2 × quadratic term (β₂)
pval_slope = nan(nx, ny);      % p-value for linear term
pval_accel = nan(nx, ny);      % p-value for acceleration term

%%%%%%%%         Loop over grid cells
for i = 1:117
    for j = 1:97
        ts = squeeze(dm(i, j, :));  % time series at (i,j)

            if (isfinite(ts(1))) 
        
            [b, ~, r] = regress(ts, X);

            XtX_inv = inv(X' * X);          
            N = size(X, 1);                 
            p = size(X, 2);                
            dof = N - p;        
       
            % Coefficients
            slope = b(2);                 
            accel = 2 * b(3);             

            % Mean squared error
            MSE = sum(r.^2) / dof;

            % Standard errors
            se_slope = sqrt(MSE * XtX_inv(2,2));
            se_quad = sqrt(MSE * XtX_inv(3,3));

            % t-stats
            t_slope = b(2) / se_slope;
            t_quad = b(3) / se_quad;

            % Two-tailed p-values
            p_slope = 2 * (1 - tcdf(abs(t_slope), dof));
            p_quad = 2 * (1 - tcdf(abs(t_quad), dof));

            % Save results
            trend(i,j) = slope;
            acceleration(i,j) = accel;
            pval_slope(i,j) = p_slope;
            pval_accel(i,j) = p_quad;
        end
    end
end


%%%%%%%%%%%%%%%%%%%%%  Plot figure %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

figure
clf
set(gcf,'color','white')
set(gcf,'Position',[43,199,627,392]);
t = tiledlayout(1,2,'TileSpacing','Compact','Padding','compact');

nexttile
pcolor(x,y,trend'); shading interp; colormap(cmap);
caxis([-200 200]);
axis off; axis image; hold on;
plot(c1,c2,'k');
hold on;
for k = 1:length(basins)
    plot(basins(k).X, basins(k).Y, 'k');
end
stipple(x, y, (pval_slope > 0.05)', 'density',250, 'color','k', 'marker','.','markersize',3);
cb1 = colorbar('Location', 'southoutside');
text(0.03,1.,0, 'a)', 'Units', 'normalized', 'FontSize', 11, 'FontWeight', 'normal');

cb1.Label.String = 'kg m^{-2} y^{-1}';
cb1.Label.FontSize = 10;
cb1.TickDirection='out';
cb1.TickLength=0.01;
title(cb1,'Rate','FontWeight','normal','FontSize',10)
cb1.FontSize = 10.5;


nexttile
pcolor(x,y,acceleration'); shading interp; colormap(cmap);
caxis([-20 20]);
axis off; axis image; hold on;
plot(c1,c2,'k');
stipple(x, y, (pval_accel > 0.05)', 'density',250, 'color','k', 'marker','.','markersize',3);
hold on;
for k = 1:length(basins)
    plot(basins(k).X, basins(k).Y, 'k');
end

cb2 = colorbar('Location', 'southoutside');
cb2.Label.String = 'kg m^{-2} y^{-1}';
cb2.Label.FontSize = 10;
cb2.TickDirection='out';
cb2.TickLength=0.01;
cb2.Label.String = 'kg m^{-2} y^{-2}';
text(0.03,1.,0, 'b)', 'Units', 'normalized', 'FontSize', 11, 'FontWeight', 'normal');

title(cb2,'Acceleration','FontWeight','normal','FontSize',10)
cb2.FontSize = 10.5;


% % Save 
% print('Figure2', '-dpng', '-r300');

%%%%%%%%%%%%%%%%  create netcdfs 

outfile = 'AIS_GRACE_trend_accel_2002_2022.nc';

% Remove file if it already exists
if isfile(outfile)
    delete(outfile)
end

% Define dimensions
nccreate(outfile, 'x', 'Dimensions', {'x', length(x)}, 'Datatype', 'double');
nccreate(outfile, 'y', 'Dimensions', {'y', length(y)}, 'Datatype', 'double');

% Define variables
nccreate(outfile, 'trend', ...
    'Dimensions', {'x', length(x), 'y', length(y)}, ...
    'Datatype', 'double', 'DeflateLevel', 5);

nccreate(outfile, 'acceleration', ...
    'Dimensions', {'x', length(x), 'y', length(y)}, ...
    'Datatype', 'double', 'DeflateLevel', 5);

nccreate(outfile, 'pval_slope', ...
    'Dimensions', {'x', length(x), 'y', length(y)}, ...
    'Datatype', 'double', 'DeflateLevel', 5);

nccreate(outfile, 'pval_accel', ...
    'Dimensions', {'x', length(x), 'y', length(y)}, ...
    'Datatype', 'double', 'DeflateLevel', 5);

%%%%%%%%%%%%%%%  Write data  %%%%%%%%%%%%%%%%%

ncwrite(outfile, 'x', x);
ncwrite(outfile, 'y', y);

ncwrite(outfile, 'trend', trend);
ncwrite(outfile, 'acceleration', acceleration);
ncwrite(outfile, 'pval_slope', pval_slope);
ncwrite(outfile, 'pval_accel', pval_accel);

%%%%%%%%%%%%%%%  Add metadata  %%%%%%%%%%%%%%%%%

ncwriteatt(outfile, '/', 'title', 'Antarctic Ice Sheet GRACE Trend and Acceleration');
ncwriteatt(outfile, '/', 'source', 'GRAVIS-3_2002095-2023059_COSTG_0100_AIS_GRID_TUD_0003.nc');
ncwriteatt(outfile, '/', 'time_period', '2002–2022');
ncwriteatt(outfile, 'trend', 'units', 'kg m^{-2} y^{-1}');
ncwriteatt(outfile, 'acceleration', 'units', 'kg m^{-2} y^{-1}');
